// Background Service Worker for Translation
const GOOGLE_TRANSLATE_API = 'https://translate.googleapis.com/translate_a/single';
const cache = new Map();
const MAX_CACHE_SIZE = 1000;

// Translate text using Google Translate API
async function translate(text, targetLang = 'en', sourceLang = 'zh-CN') {
  // Check cache first
  const cacheKey = `${sourceLang}:${targetLang}:${text}`;
  if (cache.has(cacheKey)) {
    return cache.get(cacheKey);
  }

  try {
    const url = new URL(GOOGLE_TRANSLATE_API);
    url.searchParams.append('client', 'gtx');
    url.searchParams.append('sl', sourceLang);
    url.searchParams.append('tl', targetLang);
    url.searchParams.append('dt', 't');
    url.searchParams.append('q', text);

    const response = await fetch(url.toString());

    if (!response.ok) {
      throw new Error(`Translation failed: ${response.status}`);
    }

    const data = await response.json();

    // Parse response - format is [[["translated","original",null,null,3]]]
    let translatedText = '';
    if (data && data[0]) {
      for (const item of data[0]) {
        if (item[0]) {
          translatedText += item[0];
        }
      }
    }

    const result = translatedText || text;

    // Cache the result
    if (cache.size >= MAX_CACHE_SIZE) {
      // Remove oldest entry
      const firstKey = cache.keys().next().value;
      cache.delete(firstKey);
    }
    cache.set(cacheKey, result);

    return result;

  } catch (error) {
    console.error('Translation error:', error);
    return text; // Return original text on error
  }
}

// Listen for translation requests from content script
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'translate') {
    translate(request.text)
      .then(translatedText => {
        sendResponse({ translatedText });
      })
      .catch(error => {
        console.error('Translation error:', error);
        sendResponse({ translatedText: request.text });
      });

    // Return true to indicate async response
    return true;
  }
});

// Clear cache on browser startup
chrome.runtime.onStartup.addListener(() => {
  cache.clear();
});
