// Live Translation Content Script
let isEnabled = true;
const translatedNodes = new WeakMap();
const pendingTranslations = new Map();
let translationQueue = [];
let isProcessing = false;

// Chinese character detection regex
const chineseRegex = /[\u4e00-\u9fff\u3400-\u4dbf\u{20000}-\u{2a6df}\u{2a700}-\u{2b73f}\u{2b740}-\u{2b81f}\u{2b820}-\u{2ceaf}\uf900-\ufaff\u3300-\u33ff\ufe30-\ufe4f\uf900-\ufaff\u{2f800}-\u{2fa1f}]/u;

// Check if node contains Chinese text
function containsChinese(text) {
  return text && chineseRegex.test(text);
}

// Check if element should be translated
function shouldTranslate(node) {
  if (node.nodeType !== Node.TEXT_NODE) return false;
  if (!node.textContent || node.textContent.trim().length === 0) return false;
  if (!containsChinese(node.textContent)) return false;

  // Skip script, style, and other non-visible elements
  const parent = node.parentElement;
  if (!parent) return false;

  const tagName = parent.tagName?.toLowerCase();
  if (['script', 'style', 'noscript', 'iframe', 'object'].includes(tagName)) {
    return false;
  }

  return true;
}

// Translate text via background script
async function translateText(text) {
  return new Promise((resolve) => {
    chrome.runtime.sendMessage(
      { action: 'translate', text: text },
      (response) => {
        if (chrome.runtime.lastError) {
          console.error('Translation error:', chrome.runtime.lastError);
          resolve(text); // Return original on error
        } else {
          resolve(response?.translatedText || text);
        }
      }
    );
  });
}

// Process translation queue
async function processQueue() {
  if (isProcessing || translationQueue.length === 0) return;

  isProcessing = true;

  // Process in batches of 50 for faster throughput
  while (translationQueue.length > 0) {
    const batch = translationQueue.splice(0, 50);

    await Promise.all(batch.map(async (item) => {
      const { node, originalText } = item;

      // Skip if node no longer in document
      if (!document.contains(node)) return;

      // Check if already translated
      if (translatedNodes.has(node)) return;

      const parent = node.parentElement;
      if (!parent) return;

      // Add yellow background while translating
      const originalBg = parent.style.backgroundColor;
      const originalBorder = parent.style.borderLeft;
      parent.style.backgroundColor = '#fff3cd';
      parent.style.transition = 'background-color 0.3s, border-left 0.3s';

      const translatedText = await translateText(originalText);

      // Update node if still in document
      if (document.contains(node) && node.textContent === originalText) {
        node.textContent = translatedText;
        translatedNodes.set(node, {
          original: originalText,
          translated: translatedText
        });

        // Flash green then add permanent subtle left border
        parent.style.backgroundColor = '#d4edda';
        setTimeout(() => {
          parent.style.backgroundColor = originalBg;
          parent.style.borderLeft = '2px solid rgba(76, 175, 80, 0.3)';
          parent.style.paddingLeft = parent.style.paddingLeft || '4px';
        }, 800);
      } else {
        // Restore original if translation failed
        parent.style.backgroundColor = originalBg;
      }
    }));

    // No delay - process as fast as possible
  }

  isProcessing = false;
}

// Translate a single node
function queueNodeForTranslation(node) {
  if (!shouldTranslate(node)) return;
  if (translatedNodes.has(node)) return;

  const originalText = node.textContent.trim();

  translationQueue.push({
    node: node,
    originalText: originalText
  });

  // Instant processing - no debounce
  if (!isProcessing) {
    processQueue();
  }
}

// Scan and translate all text nodes
function scanAndTranslate(root = document.body) {
  if (!isEnabled || !root) return;

  const walker = document.createTreeWalker(
    root,
    NodeFilter.SHOW_TEXT,
    {
      acceptNode: function(node) {
        return shouldTranslate(node) ?
          NodeFilter.FILTER_ACCEPT :
          NodeFilter.FILTER_REJECT;
      }
    }
  );

  let node;
  while (node = walker.nextNode()) {
    queueNodeForTranslation(node);
  }
}

// MutationObserver for live updates
const observer = new MutationObserver((mutations) => {
  if (!isEnabled) return;

  for (const mutation of mutations) {
    // Handle added nodes
    if (mutation.type === 'childList') {
      mutation.addedNodes.forEach((node) => {
        if (node.nodeType === Node.TEXT_NODE) {
          queueNodeForTranslation(node);
        } else if (node.nodeType === Node.ELEMENT_NODE) {
          scanAndTranslate(node);
        }
      });
    }

    // Handle text changes
    if (mutation.type === 'characterData') {
      const textNode = mutation.target;
      if (shouldTranslate(textNode)) {
        // Remove from translated cache to allow re-translation
        translatedNodes.delete(textNode);
        queueNodeForTranslation(textNode);
      }
    }
  }
});

// Start observing
function startObserving() {
  observer.observe(document.body, {
    childList: true,
    subtree: true,
    characterData: true,
    characterDataOldValue: true
  });
}

// Initialize extension
function init() {
  // Check if enabled from storage
  chrome.storage.sync.get(['enabled'], (result) => {
    isEnabled = result.enabled !== false; // Default to true

    if (isEnabled) {
      // Initial scan
      scanAndTranslate();

      // Start observing for changes
      startObserving();
    }
  });
}

// Listen for enable/disable messages
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'toggle') {
    isEnabled = request.enabled;

    if (isEnabled) {
      scanAndTranslate();
      startObserving();
    } else {
      observer.disconnect();
      translationQueue = [];
    }

    sendResponse({ success: true });
  }
});

// Start when DOM is ready
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', init);
} else {
  init();
}
